/*****************************************************************************************************************
	ActivityMonitor
    Log the activity of a visitor for analyzing if the visitor is active and engaged on the VIN site.
    Activity is defined by the visitor doing any of a variety of mouse events within a time frame:
       click, keypress, scroll, and mouse-move
    Any one of these events happening within a 30 second time frame indicates the user is active.
    The activity event is logged to the database for later analysis.  
    Only the first activity within the time frame is noticed and logged.
    A gotcha is that the "mouse-move" event and "scroll" event fire even when the window is in the background and not active.
    Do not count a visitor as "active and engaged" when the visitor is mouseing over or scrolling a background window.
    Do not notice or log those events when the window is not in focus.

    The event "blur" fires when a window is put into the background.
    The event "focus" fires when a window is put into focus.
    The blur event is used to disable activity events, log any unlogged events, and stop the sequence of time-framed logging.
    The focus event is used to enable activity events, and start the sequence of time-framed logging.
    The WindowIsActive flag is managed by the blur and focus events to provide state for controlling allowed actions.

*****************************************************************************************************************/
var ActivityMonitor = {
    ActivityType: { Click: false, KeyPress: false, Scroll: false, MouseMove: false, Print: false },
    LastActivity: null,
    PingCount: 0,
    PingSeconds: 30,
    ShowStatus: false,
    DebugToConsole: false, /* true - turn on debug output to console */
    Url: null,
    PingTimer: null,
    PageID: null,
    WindowIsActive: true,

    DebugLocal(message) {
        if (this.DebugToConsole) {
            console.log("ActivityMonitor: " + message);
        }
    },

    /* notice that a "type" event has fired, and stop enabling any others */
    RegisterEventFiring: function (type, ev) {
        this.DisableActivityEvents(); 
        this.ActivityType[type] = true;
        this.LastActivity = new Date();
        if (this.ShowStatus) {
            window.status = window.status.startsWith(type) ? window.status + "." : type;
        }
    },

    ParseDate: function(dt, def) {
        var ret = new Date(dt);
        if (isNaN(ret) && typeof dt === "string" && dt.match(/\d{4}-\d{2}-\d{2}((T| )\d{2}:\d{2}:\d{2}Z?)?/)) {
            ret = new Date(dt.gsub("-", "/").gsub("T", " "));
            if (isNaN(ret)) {
                ret = new Date(dt.gsub("-", "/").gsub("T", " ").gsub("Z", " GMT"));
            }
        }
        if (isNaN(ret) && Object.isDate(def)) {
            ret = def;
        }
        return ret;
    },

    GetTimeSpan: function(a, b) {
        if (arguments.length === 1) {
            return this.GetTimeSpan(a, new Date());
        }
        a = this.ParseDate(a);
        b = this.ParseDate(b);
        var span = { "Milliseconds": Math.abs(a.getTime() - b.getTime()) };
        span.Negative = b.getTime() < a.getTime();
        span.Seconds = span.Milliseconds / 1000;
        span.Minutes = span.Seconds / 60;
        span.Hours = span.Minutes / 60;
        span.Days = span.Hours / 24;
        span.Weeks = span.Days / 7;
        span.Years = span.Days / 365;
        span.Months = span.Years * 12;
        return span;
    },

    IsIdle: function(intSeconds) {
        return (intSeconds ? intSeconds : 5.0) < this.GetTimeSpan(this.LastActivity)[intSeconds ? "Seconds" : "Minutes"];
    },

    Ping: function(ev) {
        ActivityMonitor.DebugLocal('consider sending data to server');
        if (this.IsIdle() || this.Url === null || (typeof NOAJAX !== "undefined") || (!this.ActivityType.Click && !this.ActivityType.KeyPress && !this.ActivityType.Scroll && !this.ActivityType.MouseMove)) {
            this.EnablePingTimer();
            ActivityMonitor.DebugLocal('no data sent to server: ' + this.Url);
            return;
        }
        ActivityMonitor.DebugLocal('data is being sent to server: ' + this.Url);
        var strPingID = new Date().getTime() + ":" + this.PingCount++;
        this.DisablePingTimer();
        var data = {
            PageID: this.PageID,
            PingID: strPingID,
            Click: this.ActivityType["Click"],
            KeyPress: this.ActivityType["KeyPress"],
            Scroll: this.ActivityType["Scroll"],
            MouseMove: this.ActivityType["MouseMove"],
            Print: this.ActivityType["Print"]
        };
        return $j.ajax({
            url: this.Url,
			cache: false,
            contentType: "application/json; charset=UTF-8",
            data: data,
            dataType: "json",
            method: "GET"
        }).done(function(response) {
            ActivityMonitor.PingSuccess(response);
        }).fail(function(xhr, status, error) {
            ActivityMonitor.PingFailure(xhr, status, error);
        });
    },

    PingSuccess: function(response) {
        this.ActivityType = { Click: false, KeyPress: false, Scroll: false, MouseMove: false, Print: false };
        if (this.ShowStatus) {
            window.status = this.PingCount + "~" + this.PageID;
        }
        /* stop the sequence of logging-event time-frames when the window is not active */
        if (this.WindowIsActive) {
            this.EnableActivityEvents();
            this.EnablePingTimer();
        }
    },

    PingFailure: function(xhr, status, error) {
        if (this.ShowStatus) {
            window.status = this.PingCount + "!" + this.PageID;
        }
    },

    DisablePingTimer: function() {
        if (this.PingTimer) {
            clearTimeout(this.PingTimer);
        }
        this.PingTimer = null;
    },

    EnablePingTimer: function() {
        if (this.PingTimer) {
            this.DisablePingTimer();
        }
        this.PingTimer = window.setTimeout(function() { ActivityMonitor.Ping(); }, this.PingSeconds * 1000);
    },

    OnPrintAfter: function (ev) {
        ActivityMonitor.DebugLocal('on PrintAfter');
        ActivityMonitor.RegisterEventFiring("Print", ev);
    },

    OnClick: function(ev) {
        ActivityMonitor.DebugLocal('on Click event');
        ActivityMonitor.RegisterEventFiring("Click", ev);
    },

    OnKeyPress: function(ev) {
        ActivityMonitor.DebugLocal('on KeyPress event');
        ActivityMonitor.RegisterEventFiring("KeyPress", ev);
    },

    OnScroll: function(ev) {
        ActivityMonitor.DebugLocal('on Scroll event');
        ActivityMonitor.RegisterEventFiring("Scroll", ev);
    },

    OnMouseMove: function(ev) {
        ActivityMonitor.DebugLocal('on MouseMove event');
        ActivityMonitor.RegisterEventFiring("MouseMove", ev);
    },

    /* stop noticing and logging activity events when window goes out of focus */
    OnBlur: function(ev) {
        ActivityMonitor.DebugLocal('on Blur event');
        ActivityMonitor.DisableActivityEvents(); /* window must be active for this disable to work */
        ActivityMonitor.DisablePingTimer(); 
        ActivityMonitor.WindowIsActive = false;  /* prevent PingSucess from enabling activity-events and PingTimer */
        ActivityMonitor.Ping(ev);                /* log any unsent events */
    },

    /* start noticing and logging activity events when window comes into focus */
    OnFocus: function(ev) {
        ActivityMonitor.DebugLocal('on Focus event');
        ActivityMonitor.WindowIsActive = true;      /* allows activity-events enabling now that the window is in focus */
        ActivityMonitor.EnableActivityEvents();     /* window must be active for this to work */
        ActivityMonitor.EnablePingTimer(); 
    },

    EnableNoticingActiveWindow: function() {
        $j(window).blur(this.OnBlur);
        $j(window).focus(this.OnFocus);
    },

    EnableActivityEvents: function() {
        ActivityMonitor.DebugLocal('call EnableActivityEvents');
        if (!ActivityMonitor.WindowIsActive) {
            /* prevent the enabling of activity events while window is not active */
            ActivityMonitor.DebugLocal('inactive window.  event state not changed.');
        }
        ActivityMonitor.DebugLocal('events enabled');
        $j(window).mousemove(this.OnMouseMove);
        $j(window).keypress(this.OnKeyPress);
        $j(window).click(this.OnClick);
        $j(window).scroll(this.OnScroll);
        $j(document).scroll(this.OnScroll);
        $j("#contentBody").scroll(this.OnScroll);  /* for instance in Search */
        $j(window).on("touchstart", this.OnClick);
        window.onafterprint = ActivityMonitor.OnPrintAfter; /* look for jquery syntax for consistency */
    },

    DisableActivityEvents: function() {
        ActivityMonitor.DebugLocal('call DisableActivityEvents');
        if (!ActivityMonitor.WindowIsActive) {
            /* prevent the disabling of activity events while window is not active */
            ActivityMonitor.DebugLocal('inactive window.  event state not changed.');
        }
        ActivityMonitor.DebugLocal('events disabled');
        $j(window).unbind("mousemove", this.OnMouseMove);
        $j(window).unbind("keypress", this.OnKeyPress);
        $j(window).unbind("click", this.OnClick);
        $j(window).unbind("scroll", this.OnScroll);
        $j(document).unbind("scroll", this.OnScroll);
        $j("#contentBody").unbind("scroll", this.OnScroll);  /* for instance in Search */
        $j(window).off("touchstart", this.OnClick);
    },

    Initialize: function(strUrl, strPageID, intPingSeconds) {
        console.log('ActivityMonitor.js - ActivityMonitor.Initialize()  Server: ' + strUrl + '  PageID: ' + strPageID + '  Interval: ' + intPingSeconds);
        this.Url = strUrl;
        this.PageID = strPageID;
        this.PingSeconds = intPingSeconds;
        if (window.jQuery && typeof $j !== "undefined") {
            $j(document).ready(function() {
                ActivityMonitor.WindowIsActive = true;
                ActivityMonitor.LastActivity = new Date();
                ActivityMonitor.EnableNoticingActiveWindow();
                ActivityMonitor.EnablePingTimer();
                ActivityMonitor.EnableActivityEvents();
            });
        }
    }
};

/* this is below here because it didn't work inside EnableActivityEvents */
/* will continue to research */
if (window.matchMedia) {
    var mediaQueryList = window.matchMedia('print');
    mediaQueryList.addListener(function (mql) {
        if (!mql.matches) {
            ActivityMonitor.OnPrintAfter();
        }
    });
}